﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Serialization;
using UnityEngine;

//SettingsManager is a singleton class responsible for managing all settings in the game
public class SettingsManager : MonoBehaviour
{
    public Dictionary<Constants.Settings, object> Settings { get; private set; }

    private static Dictionary<Constants.Settings, object> _DefaultSettings = new Dictionary<Constants.Settings, object>();

    //Singleton
    private static SettingsManager _Instance;
    public static SettingsManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<SettingsManager>();
            }

            return _Instance;
        }
    }

    /// <summary>
    /// Initializes the SettingsManager by reading the file and setting up the default values
    /// </summary>
    /// <returns>Was the initialization successful?</returns>
    public bool Initialize()
    {
        try
        {
            GetDefaultSettings();   //Fill the DefaultSettings dictionary

            if (File.Exists(Constants.SettingsXMLFilePath))
            {
                Settings = ReadSettingsFile();  //File exists, let's read the settings

                if (Settings == null)    //Null was returned so something went wrong, let's delete the settings file and start again with the defaults
                {
                    File.Delete(Constants.SettingsXMLFilePath);
                    CreateNewSettingsFile();
                }
            }

            else
            {
                CreateNewSettingsFile();    //No file exists, let's create one with the defaults
            }

            return true;
        }

        catch(Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }

    /// <summary>
    /// Reads the settings file, returning a parsed dictionary of settings and their values
    /// </summary>
    /// <returns>A parsed dictionary of settings and their values</returns>
    private Dictionary<Constants.Settings, object> ReadSettingsFile()
    {
        try
        {
            if (VerifyAllSettingsPresent()) //All settings have entries in the file
            {
                Dictionary<Constants.Settings, object> retDict = new Dictionary<Constants.Settings, object>();
                XDocument settingsXML = XDocument.Load(Constants.SettingsXMLFilePath);

                foreach (var setting in settingsXML.Descendants("Setting"))
                {
                    //Add the name of the setting parsed into an enum and its value into the dictionary
                    retDict.Add((Constants.Settings)Enum.Parse(typeof(Constants.Settings), setting.Attribute("Name").Value.ToString(), true), setting.Value);
                }

                return retDict;
            }

            else
            {
                return null;    //We're missing settings so something's wrong, return null
            }
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when reading settings file, returning false and creating new file. The exception is: " + ex);
            return null;
        }
    }

    /// <summary>
    /// Verifies that the settings file contains an entry for each setting
    /// </summary>
    /// <returns>Did the file contain an entry for each setting?</returns>
    private bool VerifyAllSettingsPresent()
    {
        XDocument settingsXML = XDocument.Load(Constants.SettingsXMLFilePath);

        var settingsXMLList = (from ele in settingsXML.Descendants("Setting")
                               select ele).ToList();    //Get all the setting elements

        foreach (string enumVal in Enum.GetNames(typeof(Constants.Settings)))
        {
            //Loop through all values in the enum
            bool foundEnumValue = false;

            foreach (XElement settingElem in settingsXMLList)
            {
                //Loop through all the settings
                if (settingElem.Attribute("Name").Value == enumVal)
                {
                    //This setting has an entry, found it, so break from the settings loop and move onto the next enum entry
                    foundEnumValue = true;
                    break;
                }
            }

            if (!foundEnumValue)
            {
                //We didn't find it so we have missing settings, return false
                return false;
            }

            foundEnumValue = false;
        }

        //We didn't return in the loop so we've got all settings, return true
        return true;
    }

    /// <summary>
    /// Populates the DefaultSettings dictionary with default values 
    /// </summary>
    private static void GetDefaultSettings()
    {
        //Create the default settings dictionary
        _DefaultSettings = new Dictionary<Constants.Settings, object>();
        _DefaultSettings.Add(Constants.Settings.MusicVolume, 0.5f);
        _DefaultSettings.Add(Constants.Settings.SFXVolume, 1.0f);
        _DefaultSettings.Add(Constants.Settings.VehicleVolume, 0.5f);
    }

    /// <summary>
    /// Creates a new settings file populated with all the default values
    /// </summary>
    private void CreateNewSettingsFile()
    {
        File.Delete(Constants.SettingsXMLFilePath); //Delete one if it exists

        XDocument settingsXML = new XDocument(
                                            new XDeclaration("1.0", "utf-8", null),
                                            new XElement("Root",
                                            from ent in _DefaultSettings
                                            select new XElement("Setting", new XAttribute("Name", ent.Key.ToString()), ent.Value)
                                            ));

        //Create a new document by looping through each default settings entry

        settingsXML.Save(Constants.SettingsXMLFilePath);


        //The file has been made and saved, let's make the Settings dictionary with the defaults
        Settings = new Dictionary<Constants.Settings, object>();
        foreach (var defaultSetting in _DefaultSettings)
        {
            Settings[defaultSetting.Key] = defaultSetting.Value;
        }
    }

    /// <summary>
    /// Writes the settings dictionary to the Settings XML file
    /// </summary>
    public void WriteSettings()
    {
        XDocument settingsXML = new XDocument(
                                        new XDeclaration("1.0", "utf-8", "yes"),
                                        new XElement("Root", Settings.Select(val => new XElement("Setting", new XAttribute("Name", val.Key), val.Value))
                                            ));

        //Create a new document by looping through each default settings entry then save it

        settingsXML.Save(Constants.SettingsXMLFilePath);
    }
}
